<?php
session_start();
require_once 'php/config.php';

if (!is_logged_in()) {
    header("Location: login.php");
    exit;
}

$user_id = $_SESSION['user']['id'];
$assessment_id = $_GET['id'] ?? 0;

/* Pagination setup */
$limit = 5; // questions per page
$page = max(1, (int)($_GET['page'] ?? 1));
$offset = ($page - 1) * $limit;

/* Fetch result */
$resStmt = $pdo->prepare("
SELECT r.score, r.total_questions, a.assessment_title
FROM results r
JOIN assessments a ON r.assessment_id = a.id
WHERE r.user_id = :uid AND r.assessment_id = :aid
LIMIT 1
");
$resStmt->execute([
    'uid' => $user_id,
    'aid' => $assessment_id
]);
$res = $resStmt->fetch(PDO::FETCH_ASSOC);
if (!$res) die("❌ Result not found.");

/* Safe percentage */
$percentage = $res['total_questions'] > 0
    ? round(($res['score'] / $res['total_questions']) * 100)
    : 0;

/* Load all questions for stats and splitting */
$qStmt = $pdo->prepare("
SELECT id, question, option_a, option_b, option_c, option_d,
       correct_option, explanation_correct, explanation_wrong
FROM questions
WHERE assessment_id = :aid
ORDER BY id ASC
");
$qStmt->execute(['aid' => $assessment_id]);
$allQuestions = $qStmt->fetchAll(PDO::FETCH_ASSOC);

/* Load user answers */
$userAnswers = $_SESSION['review_answers'][$assessment_id] ?? [];

/* Split passed & failed */
$passed = $failed = [];
$unanswered = 0;

foreach ($allQuestions as $q) {
    $ans = $userAnswers[$q['id']] ?? '';
    if ($ans === '') $unanswered++;
    if (strtolower($ans) === strtolower($q['correct_option'])) {
        $passed[] = $q;
    } else {
        $failed[] = $q;
    }
}

/* Combine passed and failed for pagination */
$allResults = array_merge($passed, $failed);
$total_questions = count($allResults);
$total_pages = ceil($total_questions / $limit);

/* Slice questions for current page */
$currentQuestions = array_slice($allResults, $offset, $limit);
?>

<!doctype html>
<html>
<head>
<title>Assessment Result</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
.correct { border-left: 4px solid green; padding-left: 8px; }
.wrong { border-left: 4px solid red; padding-left: 8px; }
.card-body p { max-width: 600px; } /* reduce width */
</style>
</head>
<body>

<div class="container mt-5">
  <h3><?= htmlspecialchars($res['assessment_title']) ?></h3>

  <!-- Summary Stats -->
  <div class="card p-4 shadow-sm rounded-4 mb-4">
    <h4>📊 Your Result</h4>
    <p><strong>Score:</strong> <?= $res['score'] ?> / <?= $res['total_questions'] ?></p>
    <p><strong>Percentage:</strong> <?= $percentage ?>%</p>
    <p><strong>Total Correct:</strong> <?= count($passed) ?></p>
    <p><strong>Total Wrong:</strong> <?= count($failed) ?></p>
    <p><strong>Unanswered:</strong> <?= $unanswered ?></p>
    <div>
      <?php if ($percentage >= 50): ?>
        <span class="badge bg-success fs-6">PASS</span>
      <?php else: ?>
        <span class="badge bg-danger fs-6">FAIL</span>
      <?php endif; ?>
    </div>
  </div>

  <!-- Questions -->
  <?php foreach ($currentQuestions as $i => $q): 
    $userAns = $userAnswers[$q['id']] ?? '';
    $isCorrect = strtolower($userAns) === strtolower($q['correct_option']);
    $cardClass = $isCorrect ? 'correct' : 'wrong';
  ?>
  <div class="card mb-3 <?= $cardClass ?> p-3 shadow-sm">
    <p><strong>Q<?= $offset + $i + 1 ?>.</strong> <?= htmlspecialchars($q['question']) ?></p>
    <?php foreach (['a','b','c','d'] as $opt): 
      $class = '';
      if ($opt === $q['correct_option']) $class = 'text-success fw-bold';
      if ($opt === $userAns && !$isCorrect) $class = 'text-danger fw-bold';
    ?>
    <p class="<?= $class ?>"><?= strtoupper($opt) ?>. <?= htmlspecialchars($q['option_'.$opt]) ?></p>
    <?php endforeach; ?>
    <p><strong>Your Answer:</strong> <?= $userAns ? strtoupper($userAns) : 'Not Answered' ?></p>
    <p><strong>Correct Answer:</strong> <?= strtoupper($q['correct_option']) ?></p>
    <?php if ($isCorrect && $q['explanation_correct']): ?>
      <div class="alert alert-success">
        <strong>Explanation:</strong><br><?= nl2br(htmlspecialchars($q['explanation_correct'])) ?>
      </div>
    <?php elseif (!$isCorrect && $q['explanation_wrong']): ?>
      <div class="alert alert-warning">
        <strong>Why wrong / Failed:</strong><br><?= nl2br(htmlspecialchars($q['explanation_wrong'])) ?>
      </div>
    <?php endif; ?>
  </div>
  <?php endforeach; ?>

  <!-- Pagination -->
  <nav aria-label="Question pagination">
    <ul class="pagination justify-content-center">
      <?php if ($page > 1): ?>
        <li class="page-item">
          <a class="page-link" href="?id=<?= $assessment_id ?>&page=<?= $page-1 ?>">⬅ Previous</a>
        </li>
      <?php endif; ?>
      <?php for ($p=1; $p<=$total_pages; $p++): ?>
        <li class="page-item <?= $p == $page ? 'active' : '' ?>">
          <a class="page-link" href="?id=<?= $assessment_id ?>&page=<?= $p ?>"><?= $p ?></a>
        </li>
      <?php endfor; ?>
      <?php if ($page < $total_pages): ?>
        <li class="page-item">
          <a class="page-link" href="?id=<?= $assessment_id ?>&page=<?= $page+1 ?>">Next ➡</a>
        </li>
      <?php endif; ?>
    </ul>
  </nav>

</div>
</body>
</html>
