// =======================
// GLOBAL STATE
// =======================
let selectedCategoryId = null;
let currentPage = 1;
const COURSES_PER_PAGE = 4;

let modulesList = [];
let currentModuleIndex = 0;
let completedModules = new Set(); // track completed module IDs
let userSubscriptions = {}; // track active subscriptions

// =======================
// Load Categories
// =======================
fetch("php/fetch_categories.php")
.then(res => res.json())
.then(categories => {
    const wrapper = document.getElementById("categories");
    categories.forEach(cat => {
        const div = document.createElement("div");
        div.classList.add("category-item");
        div.textContent = cat.name;
        div.dataset.id = cat.id;

        div.onclick = () => {
            document.querySelectorAll(".category-item").forEach(el => el.classList.remove("selected"));
            div.classList.add("selected");
            selectedCategoryId = cat.id;
            fetchCourses(cat.id);
        };

        wrapper.appendChild(div);
    });
});

// =======================
// Fetch Courses
// =======================
function fetchCourses(category_id, page=1) {
    currentPage = page;
    fetch("php/fetch_courses.php", {
        method:"POST",
        headers: {"Content-Type":"application/x-www-form-urlencoded"},
        body: `category_id=${category_id}&page=${page}&limit=${COURSES_PER_PAGE}`
    })
    .then(res => res.json())
    .then(data => {
        const wrapper = document.getElementById("courses");
        wrapper.innerHTML = "";

        data.courses.forEach(course => {
            const div = document.createElement("div");
            div.classList.add("course-item");

            div.innerHTML = `
                <strong>${course.name}</strong> <br>
                Level: ${course.level} | Price: ₦${course.price.toLocaleString()} <br>
                <button onclick="loadModules(${course.id})">View Modules</button>
                <div id="modulesBox_${course.id}"></div>
                <div id="certificateBox_${course.id}"></div>
            `;
            wrapper.appendChild(div);
        });

        renderPagination(data.total_pages);
    });
}

// // =======================
// // Load Modules
// // =======================
// function loadModules(courseId) {
//     const box = document.getElementById(`modulesBox_${courseId}`);
//     if(!box) return;

//     box.innerHTML = "Loading modules...";

//     fetch("php/fetch_modules.php", {
//         method: "POST",
//         headers: { "Content-Type": "application/x-www-form-urlencoded" },
//         body: "course_id=" + courseId
//     })
//     .then(res => res.json())
//     .then(modules => {
//         box.innerHTML = "";

//         if(modules.length === 0) {
//             box.innerHTML = "No modules found for this course.";
//             return;
//         }

//         // set global modules
//         modulesList = modules.sort((a,b) => a.module_order - b.module_order);
//         currentModuleIndex = 0;

//         modules.forEach((m, i) => {
//             const lock = m.is_free == 1 ? "🆓 Free" : "🔒 Locked";

//             box.innerHTML += `
//                 <div class="module-item" id="module_${m.id}">
//                     ${i+1}. ${m.title} - ${lock} 
//                     <button onclick="playModule(${m.id},${m.is_free},${m.course_id})">Play</button>
//                 </div>
//             `;
//         });

//         // If user already subscribed, allow access to paid modules
//         checkSubscription(courseId);
//     })
//     .catch(err => {
//         console.log(err);
//         box.innerHTML = "Error loading modules";
//     });
// }

// =======================
// Load Modules with Submodules
// =======================
// =======================
// Load Modules with Collapsible Submodules
// =======================
// =======================
// Load Modules with Full Collapsible Submodules (Udemy-style)
// =======================
function loadModules(courseId) {
    const box = document.getElementById(`modulesBox_${courseId}`);
    if (!box) return;

    box.innerHTML = "Loading modules...";

    fetch("php/fetch_modules.php", {
        method: "POST",
        headers: { "Content-Type": "application/x-www-form-urlencoded" },
        body: "course_id=" + courseId
    })
    .then(res => res.json())
    .then(modules => {
        box.innerHTML = "";

        if (modules.length === 0) {
            box.innerHTML = "No modules found for this course.";
            return;
        }

        // Sort modules by module_order
        modules.sort((a, b) => a.module_order - b.module_order);

        // hide suscription button when already paid
if(userSubscriptions[courseId]){
   document.querySelector("#subscribeBtn_"+courseId)?.remove();
}

        // Map modules by parent_id
        const moduleMap = {};
        modules.forEach(m => {
            const parent = m.parent_id || 0;
            if (!moduleMap[parent]) moduleMap[parent] = [];
            moduleMap[parent].push(m);
        });

        // Recursive function to render modules
        function renderModuleList(parentId = 0, level = 0) {
            if (!moduleMap[parentId]) return "";

            let html = "";
            moduleMap[parentId].forEach((m, i) => {
                const lock = m.is_free == 1 ? "🆓 Free" : "🔒 Locked";
                const padding = level * 20;

                // Check if module has submodules
                const hasSubmodules = moduleMap[m.id] && moduleMap[m.id].length > 0;
                // const submoduleToggle = hasSubmodules
                //     ? `<button class="toggle-btn" onclick="toggleSubmodulesUdemy(${m.id})">+</button>`
                //     : "";
                const submoduleToggle = hasSubmodules
? `<span class="arrow" onclick="toggleSubmodulesUdemy(${m.id})">▶</span>`
: "";


                html += `
                    <div class="module-item" style="padding-left:${padding}px;">
                        <div style="display:flex; align-items:center; justify-content:space-between;">
                            <div>
                                ${submoduleToggle} 
                                ${level === 0 ? i + 1 + ". " : "- "} ${m.title} - ${lock}
                            </div>
                            <button onclick="playModule(${m.id}, ${m.is_free}, ${m.course_id})">Preview</button>
                        </div>
                    </div>
                    <div id="sub_${m.id}" class="submodules" style="display:none; margin-top:5px;">
                        ${renderModuleList(m.id, level + 1)}
                    </div>
                `;
            });

            return html;
        }

        box.innerHTML = renderModuleList();
        modulesList = modules;
        currentModuleIndex = 0;

        checkSubscription(courseId);
    })
    .catch(err => {
        console.error(err);
        box.innerHTML = "Error loading modules";
    });
}

// =======================
// Toggle Submodules Udemy-style
// =======================
// function toggleSubmodulesUdemy(moduleId) {
//     const allSubmodules = document.querySelectorAll(".submodules");
//     const currentSub = document.getElementById(`sub_${moduleId}`);

//     // Collapse all other submodules
//     allSubmodules.forEach(sub => {
//         if (sub !== currentSub) {
//             sub.style.display = "none";
//             const btn = sub.previousElementSibling.querySelector(".toggle-btn");
//             if (btn) btn.textContent = "+";
//         }
//     });

//     // Toggle current module
//     const isVisible = currentSub.style.display !== "none";
//     currentSub.style.display = isVisible ? "none" : "block";

//     const currentBtn = currentSub.previousElementSibling.querySelector(".toggle-btn");
//     if (currentBtn) currentBtn.textContent = isVisible ? "+" : "-";
// }

function toggleSubmodulesUdemy(moduleId) {
    const allSubmodules = document.querySelectorAll(".submodules");
    const allArrows = document.querySelectorAll(".arrow");

    const currentSub = document.getElementById(`sub_${moduleId}`);
    const currentArrow = currentSub.previousElementSibling.querySelector(".arrow");

    // Close all others
    allSubmodules.forEach(sub => {
        if (sub !== currentSub) sub.style.display = "none";
    });

    allArrows.forEach(arr => {
        if (arr !== currentArrow) arr.classList.remove("rotate");
    });

    // Toggle current
    const isOpen = currentSub.style.display === "block";

    currentSub.style.display = isOpen ? "none" : "block";
    currentArrow.classList.toggle("rotate", !isOpen);
}


// =======================
// Play Module
// =======================
function playModule(id, is_free, course_id){
    const module = modulesList.find(m => m.id == id);
    if(!module) return;

    // check access
    if(is_free !== 1 && !userSubscriptions[course_id]){
        showSubscribePrompt(course_id);
        return;
    }

    currentModuleIndex = modulesList.indexOf(module);

    const video = document.getElementById("mainVideo");
    video.src = `courses/${module.video_file}`;
    video.play();
    highlightCurrentModule();

    video.onended = () => {
        completedModules.add(module.id);
        checkCourseCompletion(course_id);

        // Automatically play next module (or trigger subscription)
        playNextModuleOrSubscribe();
    };
}

// =======================
// Play next module or trigger subscription if locked
// =======================
function playNextModuleOrSubscribe(){
    const nextIndex = currentModuleIndex + 1;

    if(nextIndex >= modulesList.length){
        alert("🎉 You have finished all modules!");
        return;
    }

    const nextModule = modulesList[nextIndex];

    if(nextModule.is_free == 1 || userSubscriptions[nextModule.course_id]){
        currentModuleIndex = nextIndex;
        playModuleByIndex(currentModuleIndex);
    } else {
        // stop and ask user to subscribe
        showSubscribePrompt(nextModule.course_id);
    }
}

// =======================
// Play Module by Index
// =======================
function playModuleByIndex(index){
    const module = modulesList[index];
    if(!module) return;
    playModule(module.id, module.is_free, module.course_id);
}

// =======================
// Next Button
// =======================
const nextBtn = document.getElementById("nextModuleBtn");
nextBtn.onclick = playNextModuleOrSubscribe;


// =======================
// Highlight Current Module
// =======================
function highlightCurrentModule() {
    modulesList.forEach(m => {
        const el = document.getElementById(`module_${m.id}`);
        if(el){
            el.style.background = '';
            el.style.color = '';
        }
    });

    if(modulesList[currentModuleIndex]){
        const el = document.getElementById(`module_${modulesList[currentModuleIndex].id}`);
        el.style.background = '#0d6efd';
        el.style.color = '#fff';
    }
}

// =======================
// Check Subscription
// =======================
function checkSubscription(courseId){
    fetch("php/check_subscription.php", {
        method:"POST",
        headers: {"Content-Type":"application/x-www-form-urlencoded"},
        body: `course_id=${courseId}`
    })
    .then(res => res.json())
    .then(data => {
        if(data.active){
            userSubscriptions[courseId] = true;
        }
    });
}

// =======================
// Check Course Completion for Certificate
// =======================
function checkCourseCompletion(courseId){
    const allModuleIds = modulesList.map(m => m.id);
    const completedAll = allModuleIds.every(id => completedModules.has(id));

    const certBox = document.getElementById(`certificateBox_${courseId}`);
    if(completedAll){
        certBox.innerHTML = `
            <button onclick="generateCertificate(${courseId})" style="
                margin-top:10px; padding:10px 15px; 
                background:green; color:white; border:none; border-radius:5px;">
                🎓 Generate Certificate
            </button>
        `;
    }
}

// =======================
// Generate Certificate
// =======================
function generateCertificate(courseId){
    fetch("php/generate_certificate.php", {
        method:"POST",
        headers: {"Content-Type":"application/x-www-form-urlencoded"},
        body: `course_id=${courseId}`
    })
    .then(res => res.blob())
    .then(blob => {
        // download the PDF certificate
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement("a");
        a.href = url;
        a.download = `certificate_course_${courseId}.pdf`;
        document.body.appendChild(a);
        a.click();
        a.remove();
    })
    .catch(err => alert("Error generating certificate"));
}

// render pagination
function renderPagination(totalPages){
    const wrapper = document.getElementById("pagination");
    if(!wrapper) return;

    wrapper.innerHTML = "";

    for(let i=1;i<=totalPages;i++){
        const btn = document.createElement("button");
        btn.textContent = i;
        btn.onclick = ()=> fetchCourses(selectedCategoryId, i);
        wrapper.appendChild(btn);
    }
}


// =======================
// Subscription Modal
// =======================
function showSubscribePrompt(courseId) {
    const box = document.getElementById("subscriptionForm");
    box.style.display = "flex";

    fetch("php/get_course.php", {
        method: "POST",
        headers: { "Content-Type": "application/x-www-form-urlencoded" },
        body: `course_id=${courseId}`
    })
    // .then(res => res.json())
    .then(res => res.text())
.then(data => {
    console.log("RAW RESPONSE:", data);
    return JSON.parse(data);
})

    .then(course => {
        if (!course || !course.price || !course.name) {
            alert("Unable to load course info.");
            return;
        }

        box.innerHTML = `
            <div style="
                background:#fff; padding:20px; border-radius:10px; 
                max-width:450px; margin:auto; text-align:left;">
                <h3 style="text-align:center;">Subscribe to Course</h3>
                <h5 style="font-weight:600; margin:10px 0;">
                    📘 ${course.name}
                </h5>

                <form id="subForm">
                    <input type="hidden" name="course_id" value="${courseId}">
                    <input type="hidden" name="amount" id="planAmount">
                    <input type="hidden" name="payment_method" id="paymentMethod" value="paystack">

                    <label>Email:</label>
                    <input type="email" name="email" required style="width:100%; padding:5px; margin-bottom:10px;">

                    <label>Subscription Plan:</label>
                    <select id="planSelect" required style="width:100%; padding:5px; margin-bottom:10px;">
                        <option value="regular">Regular</option>
                        <option value="premium">Premium</option>
                        <option value="daily">Daily</option>
                        <option value="weekly">Weekly</option>
                        <option value="monthly">Monthly</option>
                        <option value="yearly">Yearly</option>
                    </select>

                    <div id="planNumberDiv" style="display:none; margin-bottom:10px;">
                        <label>Number of units:</label>
                        <input type="number" id="planNumber" min="1" value="1" style="width:100%; padding:5px;">
                    </div>

                    <p id="priceDisplay" style="margin-top:10px; font-weight:bold;"></p>

                    <label>Payment Method:</label>
                    <select id="paymentMethodSelect" style="width:100%; padding:5px; margin-bottom:10px;">
                        <option value="paystack">Pay Online (Paystack)</option>
                        <option value="manual">Manual Bank Transfer</option>
                    </select>

                    <div id="manualInfo" style="display:none; margin-bottom:10px;"></div>

                    <div style="text-align:center; margin-top:15px;">
                        <button type="submit" style="
                            padding:10px 15px; background:#007bff; color:white; border:none; border-radius:5px; margin-right:10px;">
                            Subscribe Now
                        </button>
                        <button type="button" id="closeForm" style="
                            padding:10px 15px; background:#ccc; color:black; border:none; border-radius:5px;">
                            Close
                        </button>
                    </div>
                </form>
            </div>
        `;

        // initialize subscription modal behavior
        initializeSubscriptionForm(course);
    });
}

// =======================
// Initialize Subscription Form
// =======================
function initializeSubscriptionForm(course){
    const form = document.getElementById("subForm");
    const planSelect = document.getElementById("planSelect");
    const planNumberDiv = document.getElementById("planNumberDiv");
    const planNumber = document.getElementById("planNumber");
    const amountField = document.getElementById("planAmount");
    const priceDisplay = document.getElementById("priceDisplay");
    const paymentMethodSelect = document.getElementById("paymentMethodSelect");
    const paymentMethodField = document.getElementById("paymentMethod");
    const manualInfo = document.getElementById("manualInfo");
    const closeFormBtn = document.getElementById("closeForm");
    const box = document.getElementById("subscriptionForm");

    closeFormBtn.onclick = () => box.innerHTML = "";
    closeFormBtn.onclick = () => {
    box.innerHTML = "";   // remove modal content
    box.style.display = "none"; // hide the overlay
};


    function updatePrice(){
        let amount = parseFloat(course.price);

        if(planSelect.value === "premium") amount *= 2;

        if(["daily","weekly","monthly","yearly"].includes(planSelect.value)){
            planNumberDiv.style.display = "block";
            const num = parseInt(planNumber.value) || 1;
            amount *= num;
        } else {
            planNumberDiv.style.display = "none";
        }

        amountField.value = amount.toFixed(2);
        priceDisplay.textContent = "Price: ₦" + amount.toLocaleString();
    }

    updatePrice();
    planSelect.addEventListener("change", updatePrice);
    planNumber.addEventListener("input", updatePrice);

    paymentMethodSelect.addEventListener("change", () => {
        const method = paymentMethodSelect.value;
        paymentMethodField.value = method;

        if(method === "manual"){
            manualInfo.style.display = "block";
            const tempReference = 'TEMP_' + Math.floor(Math.random()*1000000+1);
            manualInfo.innerHTML = `
                <div style="padding:10px; background:#f9f9f9; border:1px solid #ddd;">
                    <strong>Bank Account Details:</strong><br>
                    Bank: GTBank<br>
                    Account Name: Your School Name<br>
                    Account Number: 0123456789<br><br>
                    <div style="background:#e6ffe6; padding:10px; border-left:4px solid green;">
                        Manual payment selected. Reference will appear after submitting.
                    </div>
                    <div style="color:green; font-size:16px; font-weight:bold; margin-top:10px;">
                        Temporary Reference: ${tempReference}
                    </div>
                    <button type="button" id="verifyManualBtn" style="display:none; margin-top:10px;">
                        I Paid by Bank Transfer
                    </button>
                </div>
            `;
        } else {
            manualInfo.style.display = "none";
            manualInfo.innerHTML = "";
        }
    });

    form.onsubmit = function(e){
        e.preventDefault();
        const email = form.email.value;
        const plan = planSelect.value;
        const numberVal = parseInt(planNumber.value) || 1;
        const payment_method = paymentMethodField.value;
        const amount = parseFloat(amountField.value);
        // const reference = 'SUB_' + Math.floor(Math.random()*1000000000+1);
        const reference = 'SUB' + Date.now(); // unique and safe


        if(form.dataset.submitted) return;
        form.dataset.submitted = true;

        fetch("php/subscribe.php", {
            method:"POST",
            headers: {"Content-Type":"application/x-www-form-urlencoded"},
            body: new URLSearchParams({
                email,
                course_id: course.id,
                plan,
                number: numberVal,
                amount,
                reference,
                payment_method
            })
        })
        .then(res => res.json())
        .then(data => {
            if(payment_method === "paystack"){

    if(!window.PaystackPop){
        alert("Paystack library not loaded");
        return;
    }

    const handler = PaystackPop.setup({
        key: 'pk_test_263fe98c1ab94e9f489c7d4c6dec6595f36d1472',
        email: email,
        amount: amount * 100,
        currency: "NGN",
        ref: reference,

        onClose: function(){
            alert("Payment window closed.");
        },

        callback: function(response){

            fetch("php/verify_payment.php", {
                method:"POST",
                headers: {"Content-Type":"application/x-www-form-urlencoded"},
                body: new URLSearchParams({
                    reference: response.reference,
                    subscription_id: data.subscription_id
                })
            })
            .then(res => res.text())
            .then(raw => {
                console.log("VERIFY RAW:", raw);
                return JSON.parse(raw);
            })
            .then(result => {

                if(result.status === "success"){
                    alert("✅ Payment successful!");

                    userSubscriptions[course.id] = true;

                    box.innerHTML = "";
                    box.style.display = "none";

                    loadModules(course.id);

//  window.location.href = "live_classes.php?course_id="+course.id;

                    setTimeout(()=>{
                    window.location.href = "live_classes.php?course_id=" + course.id;
                          },1500);
                } else {
                    alert("❌ " + result.message);
                }
            });
        }
    });

    handler.openIframe();
}
else {
                manualInfo.style.display = "block";
                manualInfo.innerHTML = `
                    <div style="padding:10px; background:#f9f9f9; border:1px solid #ddd;">
                        <strong>Bank Account Details:</strong><br>
                        Bank: GTBank<br>
                        Account Name: Your School Name<br>
                        Account Number: 0123456789<br><br>
                        <div style="background:#e6ffe6; padding:10px; border-left:4px solid green;">
                            Manual payment selected. Use the reference below when paying.
                        </div>
                        <div style="color:green; font-size:18px; font-weight:bold; margin-top:10px;">
                            Payment Reference: ${reference}
                        </div>
                        <button type="button" id="verifyManualBtn" style="margin-top:15px; padding:10px; background:#006600; color:white; border:none; border-radius:5px;"
                        data-subscription-id="${data.subscription_id}">I Paid by Bank Transfer</button>
                    </div>
                `;
                document.getElementById("verifyManualBtn").onclick = () => {
                    const subscription_id = document.getElementById("verifyManualBtn").dataset.subscriptionId;
                    fetch('php/verify_manual_payment.php', {
                        method:'POST',
                        headers: {"Content-Type":"application/x-www-form-urlencoded"},
                        body: new URLSearchParams({ subscription_id })
                    })
                    .then(res => res.json())
                    .then(result => {
                         if(data.status === "exists"){
                         alert("✅ You already subscribed to this course.");
                         userSubscriptions[course.id] = true;
                         loadModules(course.id);
                         box.innerHTML = "";
                         return;
                       }
                       else if(result.status === "success"){
                            alert("Payment verified! Subscription is now active.");
                            userSubscriptions[course.id] = true;
                            loadModules(course.id);
                            box.innerHTML = "";
                        } else {
                            alert(result.message || "Payment not verified yet.");
                        }
                    });
                };
            }
        })
        .catch(err => console.error(err));
    };
}
