<?php
/******************************************************************
 * SHOW PHP ERRORS ON BROWSER (FOR DEBUGGING)
 ******************************************************************/
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();
require_once '../php/config.php';

/******************************************************************
 * ADMIN AUTH CHECK
 ******************************************************************/
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header('Location: login.php');
    exit;
}

/******************************************************************
 * FETCH COURSES
 ******************************************************************/
$courses = $pdo->query("
    SELECT id, name 
    FROM courses 
    ORDER BY name ASC
")->fetchAll(PDO::FETCH_ASSOC);

/******************************************************************
 * HANDLE FORM SUBMISSION
 ******************************************************************/
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $course_id  = $_POST['course_id'] ?? '';
    $platform   = $_POST['platform'] ?? '';
    $start_time = $_POST['start_time'] ?? '';
    $duration   = $_POST['duration'] ?? '';
    $link       = $_POST['link'] ?? '';

    // Basic validation
    if (
        empty($course_id) ||
        empty($platform) ||
        empty($start_time) ||
        empty($duration) ||
        empty($link)
    ) {
        $error = "All fields are required.";
    } else {

        try {
            $stmt = $pdo->prepare("
                INSERT INTO course_links
                (course_id, platform, link, start_time, duration, created_at)
                VALUES (?, ?, ?, ?, ?, NOW())
            ");

            $stmt->execute([
                $course_id,
                $platform,
                $link,
                $start_time,
                $duration
            ]);

            header("Location: link_form.php?success=1");
            exit;

        } catch (PDOException $e) {
            $error = "Database Error: " . $e->getMessage();
        }
    }
}

/******************************************************************
 * FETCH EXISTING CLASS LINKS
 ******************************************************************/
$links = $pdo->query("
    SELECT cl.*, c.name AS course_name
    FROM course_links cl
    INNER JOIN courses c ON c.id = cl.course_id
    ORDER BY cl.start_time DESC
")->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Schedule Class</title>

    <!-- Bootstrap 5.3.3 -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light p-4">

<div class="container">

    <h3 class="mb-4">📅 Schedule a Class</h3>

    <?php if (!empty($_GET['success'])): ?>
        <div class="alert alert-success">
            Class scheduled successfully.
        </div>
    <?php endif; ?>

    <?php if (!empty($error)): ?>
        <div class="alert alert-danger">
            <?= htmlspecialchars($error) ?>
        </div>
    <?php endif; ?>

    <!-- ================= FORM ================= -->
    <form method="POST" class="card p-4 mb-5 shadow-sm">

        <div class="mb-3">
            <label class="form-label">Course</label>
            <select name="course_id" class="form-select" required>
                <option value="">-- Select Course --</option>
                <?php foreach ($courses as $c): ?>
                    <option value="<?= $c['id'] ?>">
                        <?= htmlspecialchars($c['name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="mb-3">
            <label class="form-label">Platform</label>
            <select name="platform" class="form-select" required>
                <option value="Zoom">Zoom</option>
                <option value="Google Meet">Google Meet</option>
            </select>
        </div>

        <div class="mb-3">
            <label class="form-label">Start Time</label>
            <input
                type="datetime-local"
                name="start_time"
                class="form-control"
                required
            >
        </div>

        <div class="mb-3">
            <label class="form-label">Duration (minutes)</label>
            <input
                type="number"
                name="duration"
                class="form-control"
                value="60"
                min="1"
                required
            >
        </div>

        <div class="mb-3">
            <label class="form-label">Meeting Link</label>
            <input
                type="url"
                name="link"
                class="form-control"
                placeholder="https://zoom.us/j/xxxxxxxx"
                required
            >
        </div>

        <button type="submit" class="btn btn-primary">
            Schedule Class
        </button>

    </form>

    <!-- ================= TABLE ================= -->
    <h4 class="mb-3">📚 Scheduled Classes</h4>

    <table class="table table-bordered table-striped">
        <thead class="table-dark">
            <tr>
                <th>#</th>
                <th>Course</th>
                <th>Platform</th>
                <th>Start Time</th>
                <th>Duration</th>
                <th>Link</th>
            </tr>
        </thead>
        <tbody>

        <?php if (count($links) === 0): ?>
            <tr>
                <td colspan="6" class="text-center text-muted">
                    No classes scheduled yet.
                </td>
            </tr>
        <?php endif; ?>

        <?php foreach ($links as $l): ?>
            <tr>
                <td><?= $l['id'] ?></td>
                <td><?= htmlspecialchars($l['course_name']) ?></td>
                <td><?= htmlspecialchars($l['platform']) ?></td>
                <td><?= date('d M Y, h:i A', strtotime($l['start_time'])) ?></td>
                <td><?= (int)$l['duration'] ?> mins</td>
                <td>
                    <a href="<?= htmlspecialchars($l['link']) ?>"
                       target="_blank"
                       class="btn btn-sm btn-success">
                        Join
                    </a>
                </td>
            </tr>
        <?php endforeach; ?>

        </tbody>
    </table>

</div>

</body>
</html>
