<?php
session_start();
require_once '../php/config.php';

if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header('Location: login.php');
    exit;
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

    $course_id = $_POST['course_id'] ?? '';
    $title = trim($_POST['title'] ?? '');
    $is_free = isset($_POST['is_free']) ? 1 : 0;
    $module_order = (int)($_POST['module_order'] ?? 1);
    $parent_id = $_POST['parent_id'] ?: null;

    $video_file = null;

    if (!empty($_FILES['video']['name'])) {
        $videoExt = strtolower(pathinfo($_FILES['video']['name'], PATHINFO_EXTENSION));
        $allowedVideos = ['mp4','avi','mov','mkv'];

        if (!in_array($videoExt, $allowedVideos)) {
            $error = 'Invalid video format.';
        } else {
            $video_file = uniqid('module_') . '.' . $videoExt;
            $videoPath = realpath(__DIR__ . '/..') . '/courses/' . $video_file;
            if (!move_uploaded_file($_FILES['video']['tmp_name'], $videoPath)) {
                $error = 'Video upload failed.';
            }
        }
    } else {
        $error = 'Please upload a video.';
    }

    if ($error === '') {
        $stmt = $pdo->prepare("INSERT INTO course_modules (course_id, title, video_file, module_order, is_free, parent_id) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$course_id, $title, $video_file, $module_order, $is_free, $parent_id]);
        $success = 'Module added successfully!';
    }
}

// Fetch courses
$courses = $pdo->query("SELECT id, name FROM courses ORDER BY name ASC")->fetchAll(PDO::FETCH_ASSOC);

// Fetch main modules
$modules = $pdo->query("SELECT id, title, course_id FROM course_modules WHERE parent_id IS NULL ORDER BY title ASC")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Add Module | Admin</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
<style>
body {
    background-color: #f4f6f8;
    font-family: 'Poppins', sans-serif;
}
.container {
    max-width: 800px;
    margin: 50px auto;
    background: #fff;
    padding: 30px;
    border-radius: 12px;
    box-shadow: 0 8px 20px rgba(0,0,0,.1);
}
h2 {
    text-align: center;
    margin-bottom: 25px;
    font-weight: 600;
    color: #0d6efd;
}
.form-label {
    font-weight: 500;
}
button {
    font-weight: 500;
}
.success {
    background:#ddffdd;
    padding:10px;
    border-left:4px solid green;
    color:#006400;
}
.error {
    background:#ffe5e5;
    padding:10px;
    border-left:4px solid red;
    color:#a30000;
}
</style>
</head>
<body>

<div class="container">
    <h2><i class="fas fa-plus-circle"></i> Add Module / Submodule</h2>

    <?php if($error): ?>
        <div class="alert alert-danger"><i class="fas fa-exclamation-triangle"></i> <?= htmlspecialchars($error) ?></div>
    <?php endif; ?>
    <?php if($success): ?>
        <div class="alert alert-success"><i class="fas fa-check-circle"></i> <?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data" class="mt-4">

        <div class="mb-3">
            <label class="form-label">Select Course</label>
            <select name="course_id" id="courseSelect" class="form-select" required>
                <option value="">-- Select Course --</option>
                <?php foreach($courses as $c): ?>
                    <option value="<?= $c['id'] ?>"><?= htmlspecialchars($c['name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="mb-3">
            <label class="form-label">Parent Module (Optional)</label>
            <select name="parent_id" id="parentSelect" class="form-select">
                <option value="">-- Main Module --</option>
                <?php foreach($modules as $m): ?>
                    <option value="<?= $m['id'] ?>" data-course="<?= $m['course_id'] ?>"><?= htmlspecialchars($m['title']) ?></option>
                <?php endforeach; ?>
            </select>
            <small class="text-muted">If you select a parent, this will be added as a submodule under it.</small>
        </div>

        <div class="mb-3">
            <label class="form-label">Module Title</label>
            <input type="text" name="title" class="form-control" placeholder="Enter module title" required>
        </div>

        <div class="mb-3">
            <label class="form-label">Upload Video</label>
            <input type="file" name="video" class="form-control" accept="video/*" required>
            <small class="text-muted">Supported formats: mp4, avi, mov, mkv</small>
        </div>

        <div class="mb-3">
            <label class="form-label">Module Order</label>
            <input type="number" name="module_order" class="form-control" placeholder="1, 2, 3…" required>
        </div>

        <div class="form-check mb-4">
            <input class="form-check-input" type="checkbox" name="is_free" id="isFree">
            <label class="form-check-label" for="isFree">
                Free Module (Preview)
            </label>
        </div>

        <button type="submit" class="btn btn-primary w-100"><i class="fas fa-plus"></i> Add Module</button>

    </form>
</div>

<script>
const courseSelect = document.getElementById('courseSelect');
const parentSelect = document.getElementById('parentSelect');

courseSelect.addEventListener('change', () => {
    const selectedCourse = courseSelect.value;
    for (let option of parentSelect.options) {
        if (!option.dataset.course) continue;
        option.style.display = option.dataset.course === selectedCourse ? 'block' : 'none';
    }
    parentSelect.value = '';
});
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
