<?php
session_start();
require_once 'php/config.php';

if (!isset($_SESSION['user'])) {
    header("Location: login.php?redirect=live_classes.php");
    exit;
}

$user = $_SESSION['user'];

// Fetch subscribed courses
$stmt = $pdo->prepare("
    SELECT c.* 
    FROM courses c
    INNER JOIN subscriptions s ON s.course_id = c.id
    WHERE s.user_id = :user_id
");
$stmt->execute([':user_id' => $user['id']]);
$courses = $stmt->fetchAll(PDO::FETCH_ASSOC);

// // Fetch Zoom/Meet links
// $linkStmt = $pdo->prepare("SELECT * FROM course_links WHERE course_id = :course_id LIMIT 1");
function getLiveClass($pdo, $course_id) {
    $stmt = $pdo->prepare("
        SELECT *
        FROM course_links
        WHERE course_id = :course_id
        ORDER BY start_time DESC
        LIMIT 1
    ");
    $stmt->execute([':course_id' => $course_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

?>

<!DOCTYPE html>
<html>
<head>
    <title>Live Classes</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" 
          href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css">
    <style>
        body { background-color: #f8f9fa; }
        .video-player { width: 100%; max-height: 400px; object-fit: cover; }
        .video-list { max-height: 600px; overflow-y: auto; }
        .video-item { cursor: pointer; padding: 10px; border-bottom: 1px solid #ddd; }
        .video-item:hover { background-color: #f1f1f1; }
        .video-item.active { background-color: #e9ecef; font-weight: bold; }
        .search-input { margin-bottom: 10px; }
        .comments { max-height: 300px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; background: #fff; }
        .comment-item { border-bottom: 1px solid #eee; padding: 5px 0; }
        .video-section { gap: 15px; }
        @media(max-width: 767px){
            .video-section { flex-direction: column; }
            .video-list { max-height: 200px; margin-top: 15px; }
        }
    </style>
</head>
<body>
<div class="container py-5">
    <h2 class="mb-4">👩‍🏫 Excellent Grade Live Classes</h2>
    <div class="alert alert-info">
        Welcome, <strong><?= htmlspecialchars($user['fullname']) ?></strong>!  
        Click a video to Learn.
    </div>

    <?php if (empty($courses)): ?>
        <div class="alert alert-warning">
            You have not subscribed to courses yet. Please subscribe to enter  classes.
        </div>
    <?php else: ?>
        <div class="d-flex video-section">
            <!-- Video Player -->
            <div class="flex-fill">
                <video id="mainVideo" class="video-player" controls poster="assets/uploads/<?= htmlspecialchars($courses[0]['thumbnail'] ?? 'default.jpg') ?>">
                    <source src="courses/<?= htmlspecialchars($courses[0]['file']) ?>" type="video/mp4">
                    Your browser does not support the video tag.
                </video>
                <h5 id="videoTitle" class="mt-2"><?= htmlspecialchars($courses[0]['name']) ?></h5>
                <p id="videoDesc"><?= htmlspecialchars($courses[0]['description']) ?></p>

                <!-- Zoom/Meet Link -->
                <?php
$liveClass = getLiveClass($pdo, $courses[0]['id']);
?>
<?php
date_default_timezone_set('Africa/Lagos');

$status = null;
$startTimestamp = null;
$endTimestamp = null;

if ($liveClass) {
    $startTimestamp = strtotime($liveClass['start_time']);
    $endTimestamp   = $startTimestamp + ($liveClass['duration'] * 60);
    $now            = time();

    if ($now < $startTimestamp) {
        $status = 'upcoming';
    } elseif ($now >= $startTimestamp && $now <= $endTimestamp) {
        $status = 'live';
    } else {
        $status = 'ended';
    }
}
?>

<?php
$upcomingStmt = $pdo->prepare("
    SELECT cl.*, c.name
    FROM course_links cl
    INNER JOIN courses c ON c.id = cl.course_id
    INNER JOIN subscriptions s ON s.course_id = c.id
    WHERE s.user_id = :uid
      AND cl.start_time > NOW()
    ORDER BY cl.start_time ASC
");
$upcomingStmt->execute(['uid' => $user['id']]);
$upcomingClasses = $upcomingStmt->fetchAll(PDO::FETCH_ASSOC);
?>

<?php if ($upcomingClasses): ?>
<div class="card mt-4">
    <div class="card-header">📅 Upcoming Live Classes</div>
    <ul class="list-group list-group-flush">
        <?php foreach ($upcomingClasses as $u): ?>
        <li class="list-group-item">
            <strong><?= htmlspecialchars($u['name']) ?></strong><br>
            🕒 <?= date("d M Y, h:i A", strtotime($u['start_time'])) ?>
        </li>
        <?php endforeach; ?>
    </ul>
</div>
<?php endif; ?>


<!-- live class box -->
<div id="liveClassBox" class="card shadow-sm my-3">
    <div class="card-body">
        <h5 class="card-title">📡 Live Class Session</h5>

        <?php if (!$liveClass): ?>
            <div class="alert alert-warning mb-0">
                No live class scheduled for this course yet.
            </div>

        <?php else: ?>
            <p><strong>Platform:</strong> <?= ucfirst($liveClass['platform']) ?></p>
            <p><strong>Starts:</strong>
                <?= date("d M Y, h:i A", strtotime($liveClass['start_time'])) ?>
            </p>
            <p><strong>Duration:</strong> <?= (int)$liveClass['duration'] ?> minutes</p>

            <!-- Countdown -->
            <div id="countdownBox" class="alert alert-warning text-center"></div>

            <!-- Join Button Logic -->
            <?php if ($status === 'upcoming'): ?>
                <button class="btn btn-secondary w-100" disabled>
                    ⏳ Class not started
                </button>

            <?php elseif ($status === 'live'): ?>
                <a href="<?= htmlspecialchars($liveClass['link']) ?>"
                   target="_blank"
                   class="btn btn-success w-100">
                    🔴 Join Live Class
                </a>

            <?php else: ?>
                <button class="btn btn-dark w-100" disabled>
                    ❌ Class Ended
                </button>
            <?php endif; ?>

            <!-- Recording -->
            <?php if ($status === 'ended' && !empty($liveClass['recording_link'])): ?>
                <a href="<?= htmlspecialchars($liveClass['recording_link']) ?>"
                   target="_blank"
                   class="btn btn-primary w-100 mt-2">
                    🎥 Watch Recording
                </a>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

                <!-- Comments Section -->
                <div class="mt-3">
                    <h6>Comments</h6>
                    <div class="comments" id="commentsList">
                        <?php
                        $commentStmt = $pdo->prepare("SELECT vc.*, u.fullname FROM video_comments vc INNER JOIN users u ON u.id = vc.user_id WHERE vc.course_id = :course_id ORDER BY vc.created_at DESC");
                        $commentStmt->execute([':course_id' => $courses[0]['id']]);
                        $comments = $commentStmt->fetchAll(PDO::FETCH_ASSOC);
                        foreach($comments as $comment):
                        ?>
                            <div class="comment-item">
                                <strong><?= htmlspecialchars($comment['fullname']) ?>:</strong>
                                <span><?= htmlspecialchars($comment['comment']) ?></span>
                                <div class="text-muted" style="font-size: 0.8em;"><?= $comment['created_at'] ?></div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <form id="commentForm" class="mt-2">
                        <input type="hidden" name="course_id" id="courseId" value="<?= $courses[0]['id'] ?>">
                        <div class="input-group">
                            <input type="text" class="form-control" name="comment" placeholder="Add a comment..." required>
                            <button class="btn btn-primary" type="submit">Post</button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Video List -->
            <div class="video-list" style="width: 300px;">
                <input type="text" id="searchVideo" class="form-control search-input" placeholder="Search videos...">
                <?php foreach ($courses as $index => $course): ?>
                    <div class="video-item <?= $index===0?'active':'' ?>" 
                         data-video="courses/<?= htmlspecialchars($course['file']) ?>"
                         data-title="<?= htmlspecialchars($course['name']) ?>"
                         data-desc="<?= htmlspecialchars($course['description']) ?>"
                         data-courseid="<?= $course['id'] ?>">
                        <?= htmlspecialchars($course['name']) ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    <?php endif; ?>
    <div class="mt-4">
        <a href="dashboard.php" class="btn btn-secondary">⬅ Back to Dashboard</a>
    </div>
</div>

<script>
<?php if ($liveClass && $status === 'upcoming'): ?>
const startTime = new Date("<?= $liveClass['start_time'] ?>").getTime();

setInterval(() => {
    const now = new Date().getTime();
    const diff = startTime - now;
    const box = document.getElementById("countdownBox");

    if (!box) return;

    if (diff <= 0) {
        box.innerHTML = "🔴 Class is LIVE now. Refresh page.";
        box.classList.replace("alert-warning", "alert-success");
        return;
    }

    const h = Math.floor(diff / (1000 * 60 * 60));
    const m = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
    const s = Math.floor((diff % (1000 * 60)) / 1000);

    box.innerHTML = `⏳ Starts in ${h}h ${m}m ${s}s`;
}, 1000);
<?php endif; ?>
</script>



<script>
document.addEventListener("DOMContentLoaded", function() {
    const videoItems = document.querySelectorAll(".video-item");
    const mainVideo = document.getElementById("mainVideo");
    const videoTitle = document.getElementById("videoTitle");
    const videoDesc = document.getElementById("videoDesc");
    const searchInput = document.getElementById("searchVideo");
    const commentsList = document.getElementById("commentsList");
    const commentForm = document.getElementById("commentForm");
    const courseIdInput = document.getElementById("courseId");

    // Video click
    videoItems.forEach(item => {
        item.addEventListener("click", () => {
            mainVideo.src = item.dataset.video;
            videoTitle.textContent = item.dataset.title;
            videoDesc.textContent = item.dataset.desc;
            mainVideo.play();
            courseIdInput.value = item.dataset.courseid;

            // Highlight active
            videoItems.forEach(i => i.classList.remove("active"));
            item.classList.add("active");

            // Reload comments
            fetch(`fetch_comments.php?course_id=${item.dataset.courseid}`)
    .then(res => res.json())
    .then(data => {
        commentsList.innerHTML = "";
        data.forEach(c => {
            let repliesHtml = '';
            if(c.replies.length > 0){
                repliesHtml = '<ul class="list-unstyled ms-3 mt-1">';
                c.replies.forEach(r => {
                    repliesHtml += `<li><small class="text-muted">Admin: ${r.reply_text} (${r.created_at})</small></li>`;
                });
                repliesHtml += '</ul>';
            }

            commentsList.innerHTML += `
                <div class="comment-item">
                    <strong>${c.fullname}:</strong> ${c.comment}
                    <div class="text-muted" style="font-size:0.8em">${c.created_at}</div>
                    ${repliesHtml}
                </div>`;
        });
    });

            // Load live class info
fetch(`fetch_live_class.php?course_id=${item.dataset.courseid}`)
    .then(res => res.json())
    .then(live => {
        const box = document.getElementById("liveClassBox");

        if (!live) {
            box.innerHTML = `
                <div class="card-body">
                    <h5 class="card-title">📡 Live Class Session</h5>
                    <div class="alert alert-warning mb-0">
                        No live class scheduled for this course yet.
                    </div>
                </div>`;
            return;
        }

        box.innerHTML = `
            <div class="card-body">
                <h5 class="card-title">📡 Live Class Session</h5>
                <p><strong>Platform:</strong> ${live.platform}</p>
                <p><strong>Starts:</strong> 
                    ${new Date(live.start_time).toLocaleString()}
                </p>
                <p><strong>Duration:</strong> ${live.duration} minutes</p>
                <a href="${live.link}" target="_blank"
                   class="btn btn-success w-100">
                    🚀 Join Live Class
                </a>
            </div>`;
    });

        });
    });

    // Search videos
    searchInput.addEventListener("input", () => {
        const query = searchInput.value.toLowerCase();
        videoItems.forEach(item => {
            if(item.dataset.title.toLowerCase().includes(query)) {
                item.style.display = "block";
            } else {
                item.style.display = "none";
            }
        });
    });

    // Post comment
    commentForm.addEventListener("submit", function(e) {
        e.preventDefault();
        const formData = new FormData(commentForm);
        fetch("post_comment.php", {
            method: "POST",
            body: formData
        }).then(res=>res.json()).then(data=>{
            if(data.status==="success"){
                // Append comment
                commentsList.innerHTML = `<div class="comment-item">
                    <strong>${data.fullname}:</strong> ${data.comment}
                    <div class="text-muted" style="font-size:0.8em">${data.created_at}</div>
                </div>` + commentsList.innerHTML;
                commentForm.reset();
            } else {
                alert(data.message);
            }
        });
    });
});
</script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
