// =======================
// GLOBAL STATE
// =======================

let selectedCategoryId = null;
let currentVideo = null;

// =======================
// PLAN PRICES
// =======================
const PLAN_PRICES = {
  regular: 3000,   // ₦3,000
  premium: 7000    // ₦7,000
};



// =======================
// Load Categories
// =======================
fetch("php/fetch_categories.php")
  .then(res => res.json())
  .then(categories => {
    const wrapper = document.getElementById("categories");

    categories.forEach(cat => {
      const div = document.createElement("div");
      div.classList.add("category-item");
      div.textContent = cat.name;

      div.onclick = () => {
        document.querySelectorAll(".category-item")
          .forEach(el => el.classList.remove("selected"));

        div.classList.add("selected"); 
        selectedCategoryId = cat.id;
        fetchCourses(cat.id);
      };

      wrapper.appendChild(div);
    });
  })
  .catch(err => console.log("Error:", err));


// =======================
// Fetch Courses
// =======================
function fetchCourses(category_id) {
  fetch("php/fetch_courses.php", {
    method: "POST",
    headers: {"Content-Type": "application/x-www-form-urlencoded"},
    body: `category_id=${category_id}`
  })
  .then(res => res.json())
  .then(courses => {
    const wrapper = document.getElementById("courses");
    wrapper.innerHTML = "";

    courses.forEach(course => {
      const div = document.createElement("div");
      div.classList.add("course-item");

      div.innerHTML = `
        <div>

          <video width="200" controls poster="assets/uploads/${course.thumbnail || 'default.jpg'}">
            <source src="courses/${course.file}" type="video/mp4">
            Your browser does not support the video tag.
          </video>

          <br>
          <strong>${course.name}</strong><br>

          Level: ${course.level}<br>
          Description: ${course.description}<br>
          Duration: ${course.duration} days<br>
          Price: ₦${course.price}<br>
        </div>

        <button class="subscribe-btn" onclick="subscribe(${course.id}, ${course.price})">
          Subscribe
        </button>
      `;

      wrapper.appendChild(div);

      const video = div.querySelector("video");
      setupVideoProtection(video, course.id);
    });

    checkCertificate();
  })
  .catch(err => console.log("Error:", err));
}

// =======================
// PREVENT VIDEO SKIPPING
// =======================

function setupVideoProtection(video, courseId) {
  if (!video) return;

  let lastTime = 0;

  // 🚫 Disable right-click (save / download)
  video.addEventListener("contextmenu", e => {
    e.preventDefault();
  });

  // 🚫 Disable download button
  video.setAttribute("controlsList", "nodownload");

  // 🚫 Prevent dragging video
  video.addEventListener("dragstart", e => {
    e.preventDefault();
  });

  // ⏸ Pause video when user switches tab
  document.addEventListener("visibilitychange", () => {
    if (document.hidden) {
      video.pause();
    }
  });

  // ⏩ Prevent skipping forward
  video.addEventListener("timeupdate", () => {
    if (video.currentTime > lastTime + 1) {
      video.currentTime = lastTime;
    } else {
      lastTime = video.currentTime;
    }
  });

  // ✅ Mark video as watched when finished
  video.addEventListener("ended", () => {
    markVideoWatched(courseId);
    checkCertificate();
  });
}

// =======================
// MARK VIDEO WATCHED
// =======================
function markVideoWatched(course_id) {
  fetch("php/mark_video_watched.php", {
    method: "POST",
    headers: {"Content-Type": "application/x-www-form-urlencoded"},
    body: `course_id=${course_id}`
  });
}

// =======================
// CHECK CERTIFICATE
// =======================
function checkCertificate() {
  if (!selectedCategoryId) return;

  fetch(`php/check_certificate.php?category_id=${selectedCategoryId}`)
    .then(res => res.json())
    .then(data => {
      if (data.completed) {
        document.getElementById("certificateBox").innerHTML = `
          <a href="php/generate_certificate.php?category_id=${selectedCategoryId}"
             class="btn btn-success">
            🎓 Download Certificate
          </a>
        `;
      }
    });
}

// =======================
// Subscribe Form
// =======================
function subscribe(course_id, coursePrice) {
  const box = document.getElementById("subscriptionForm");

  box.innerHTML = `
    <h3>Subscribe to Course</h3>
    <form id="subForm">
      <input type="hidden" name="course_id" value="${course_id}">
      <input type="hidden" name="amount" id="planAmount">
      <input type="hidden" name="payment_method" id="paymentMethod" value="paystack">

      <label>Email:</label>
      <input type="email" name="email" required>

      <label>Subscription Plan:</label>
      <select name="plan" id="planSelect" required>
        <option value="regular">Regular (Basic Access)</option>
        <option value="premium">Premium (Full Access)</option>
      </select>

      <p id="priceDisplay" style="margin-top:10px; font-weight:bold;"></p>

      <label>Duration:</label>
      <select name="duration" id="durationSelect">
        <option value="30">30 days</option>
        <option value="60">60 days</option>
        <option value="90">90 days</option>
      </select>

      <label>Payment Method:</label>
      <select name="payment_method_select" id="paymentMethodSelect">
        <option value="paystack">Pay Online (Paystack)</option>
        <option value="manual">Manual Bank Transfer</option>
      </select>

      <div id="manualInfo" style="display:none; margin-top:10px;"></div>

      <br><br>
      <button type="submit">Subscribe Now</button>
      <button type="button" id="closeForm">Close</button>
    </form>
  `;

  const planSelect = document.getElementById("planSelect");
  const amountField = document.getElementById("planAmount");
  const priceDisplay = document.getElementById("priceDisplay");
  const form = document.getElementById("subForm");
  const paymentMethodSelect = document.getElementById("paymentMethodSelect");
  const paymentMethodField = document.getElementById("paymentMethod");
  const manualInfo = document.getElementById("manualInfo");
  const closeFormBtn = document.getElementById("closeForm");

  closeFormBtn.onclick = () => box.innerHTML = "";

  function updatePrice() {
    let amount = coursePrice;
    if (planSelect.value === "premium") amount *= 2;
    amountField.value = amount;
    priceDisplay.textContent = "Price: ₦" + amount.toLocaleString();
  }
  updatePrice();
  planSelect.addEventListener("change", updatePrice);

  // Temporary reference for manual before submission
  let tempReference = 'TEMP_' + Math.floor(Math.random() * 1000000 + 1);

  // Toggle manual info
  paymentMethodSelect.addEventListener("change", () => {
    const method = paymentMethodSelect.value;
    paymentMethodField.value = method;

    if (method === "manual") {
      manualInfo.style.display = "block";
      manualInfo.innerHTML = `
        <div style="padding:10px; background:#f9f9f9; border:1px solid #ddd;">
          <strong>Bank Account Details:</strong><br>
          Bank: GTBank<br>
          Account Name: Your School Name<br>
          Account Number: 0123456789<br><br>
          <div style="background:#e6ffe6; padding:10px; border-left:4px solid green;">
            Manual payment selected. Your reference will appear after submitting.
          </div>
          <div style="color:green; font-size:18px; font-weight:bold; margin-top:10px;">
            Temporary Reference: ${tempReference}
          </div>
          <button type="button" id="verifyManualBtn" style="display:none; margin-top:10px;">
            I Paid by Bank Transfer
          </button>
        </div>
      `;
    } else {
      manualInfo.style.display = "none";
      manualInfo.innerHTML = "";
    }
  });

  // Form submission
  form.onsubmit = function(e) {
    e.preventDefault();
    const email = form.email.value;
    const plan = planSelect.value;
    const duration = document.getElementById("durationSelect").value;
    const payment_method = paymentMethodField.value;

    let amount = coursePrice;
    if (plan === "premium") amount *= 2;

    const reference = 'SUB_' + Math.floor(Math.random() * 1000000000 + 1);

    fetch("php/subscribe.php", {
      method: "POST",
      headers: {"Content-Type": "application/x-www-form-urlencoded"},
      body: new URLSearchParams({
        email,
        course_id,
        duration,
        plan,
        amount,
        reference,
        payment_method
      })
    })
    .then(res => res.json())
    .then(data => {

      if (payment_method === "paystack") {
        if (typeof PaystackPop === "undefined") {
          alert("Paystack script not loaded!");
          return;
        }
        const handler = PaystackPop.setup({
          key: 'pk_test_263fe98c1ab94e9f489c7d4c6dec6595f36d1472',
          email: email,
          amount: amount * 100,
          currency: "NGN",
          ref: reference,
          onClose: function() { alert("Payment window closed."); },
          callback: function(response) {
            fetch("php/verify_payment.php", {
              method: "POST",
              body: new URLSearchParams({
                reference: response.reference,
                subscription_id: data.subscription_id
              })
            })
            .then(res => res.json())
            .then(result => {
              if(result.status === "success") alert(result.message);
            });
          }
        });
        handler.openIframe();

      } else {
        // Update manualInfo with real reference
        manualInfo.style.display = "block";
        manualInfo.innerHTML = `
          <div style="padding:10px; background:#f9f9f9; border:1px solid #ddd;">
            <strong>Bank Account Details:</strong><br>
            Bank: GTBank<br>
            Account Name: Your School Name<br>
            Account Number: 0123456789<br><br>
            <div style="background:#e6ffe6; padding:10px; border-left:4px solid green;">
              Manual payment selected. Use the reference below when paying.
            </div>
            <div style="color:green; font-size:18px; font-weight:bold; margin-top:10px;">
              Payment Reference: ${reference}
            </div>
            <button type="button" id="verifyManualBtn" 
                style="margin-top:15px; padding:10px; background:#006600; color:white; border:none;"
                data-subscription-id="${data.subscription_id}">
                I Paid by Bank Transfer
            </button>
          </div>
        `;

        const manualBtn = document.getElementById("verifyManualBtn");
        manualBtn.onclick = () => {
          const subscription_id = manualBtn.dataset.subscriptionId;
          fetch('php/verify_manual_payment.php', {
            method: 'POST',
            headers: {"Content-Type": "application/x-www-form-urlencoded"},
            body: new URLSearchParams({ subscription_id })
          })
          .then(res => res.json())
          .then(result => {
            if(result.status === "success") {
              alert("Payment verified! Subscription is now active.");
            } else {
              alert(result.message || "Payment not verified yet.");
            }
          });
        };
      }
    })
    .catch(err => console.log(err));
  };
}

const video = document.getElementById("mainVideo");

video.addEventListener("ended", () => {
    fetch("php/mark_video_watched.php", {
        method: "POST",
        headers: {"Content-Type": "application/x-www-form-urlencoded"},
        body: new URLSearchParams({
            course_id: document.getElementById("courseId").value,
            video_id: document.getElementById("courseId").value
        })
    });
});


  

