<?php
session_start();
require_once __DIR__ . '/../php/config.php';
require_once __DIR__ . '/../includes/header.php';

// Ensure admin is logged in
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header("Location: ../login.php");
    exit;
}

/* ============================
   PAGINATION SETTINGS
============================ */
$perPage = 10;
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$page = max($page, 1);
$offset = ($page - 1) * $perPage;

/* ============================
   TOTAL RECORDS
============================ */
$countStmt = $pdo->query("SELECT COUNT(*) FROM subscriptions");
$totalRecords = (int)$countStmt->fetchColumn();
$totalPages = ceil($totalRecords / $perPage);

/* ============================
   FETCH PAGINATED DATA
============================ */
$stmt = $pdo->prepare("
    SELECT 
        s.*,
        u.fullname AS student_name,
        u.email AS student_email,
        c.name AS course_name
    FROM subscriptions s
    JOIN users u ON u.id = s.user_id
    JOIN courses c ON c.id = s.course_id
    ORDER BY s.created_at DESC
    LIMIT :limit OFFSET :offset
");

$stmt->bindValue(':limit', $perPage, PDO::PARAM_INT);
$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->execute();

$subscriptions = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Admin | Subscriptions</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">

<style>
body { background:#f8f9fa; }
.badge-paid { background:#198754; }
.badge-pending { background:#ffc107; color:#000; }
.badge-failed { background:#dc3545; }
</style>
</head>

<body>
<div class="container py-5">

    <h2 class="mb-4">📊 Subscription Monitoring</h2>

    <?php if(empty($subscriptions)): ?>
        <div class="alert alert-warning">No subscriptions found.</div>
    <?php else: ?>

    <div class="table-responsive">
        <table class="table table-bordered table-hover bg-white shadow-sm">
            <thead class="table-dark">
                <tr>
                    <th>#</th>
                    <th>Student</th>
                    <th>Email</th>
                    <th>Course</th>
                    <th>Plan</th>
                    <th>Amount</th>
                    <th>Status</th>
                    <th>Method</th>
                    <th>Start</th>
                    <th>End</th>
                    <th>Subscribed</th>
                </tr>
            </thead>
            <tbody>
            <?php foreach ($subscriptions as $index => $sub): ?>
                <tr>
                    <td><?= $offset + $index + 1 ?></td>
                    <td><?= htmlspecialchars($sub['student_name']?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($sub['student_email']?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($sub['course_name']?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= htmlspecialchars($sub['plan']?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    <td>₦<?= number_format($sub['amount'], 2) ?></td>
                    <td>
                        <?php if ($sub['payment_status'] === 'paid'): ?>
                            <span class="badge badge-paid">Paid</span>
                        <?php elseif ($sub['payment_status'] === 'pending'): ?>
                            <span class="badge badge-pending">Pending</span>
                        <?php else: ?>
                            <span class="badge badge-failed">Failed</span>
                        <?php endif; ?>
                    </td>
                    <td><?= htmlspecialchars($sub['payment_method']?? '', ENT_QUOTES, 'UTF-8') ?></td>
                    <td><?= $sub['start_date'] ? date('d M Y', strtotime($sub['start_date'])) : '-' ?></td>
                    <td><?= $sub['end_date'] ? date('d M Y', strtotime($sub['end_date'])) : '-' ?></td>
                    <td><?= date('d M Y, h:i A', strtotime($sub['created_at'])) ?></td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>

    <!-- PAGINATION -->
    <nav class="mt-4">
        <ul class="pagination justify-content-center">
            <?php if ($page > 1): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?= $page - 1 ?>">Previous</a>
                </li>
            <?php endif; ?>

            <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                <li class="page-item <?= ($i === $page) ? 'active' : '' ?>">
                    <a class="page-link" href="?page=<?= $i ?>"><?= $i ?></a>
                </li>
            <?php endfor; ?>

            <?php if ($page < $totalPages): ?>
                <li class="page-item">
                    <a class="page-link" href="?page=<?= $page + 1 ?>">Next</a>
                </li>
            <?php endif; ?>
        </ul>
    </nav>

    <?php endif; ?>

    <a href="dashboard.php" class="btn btn-secondary mt-3">⬅ Back to Admin Dashboard</a>

</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
