<?php
require_once __DIR__ . '/../php/config.php';
require_once __DIR__ . '/../includes/header.php';

// Admin protection
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header('Location: ../login.php');
    exit;
}

// Handle AJAX reply submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    $comment_id = $_POST['comment_id'];
    $reply_text = trim($_POST['reply_text']);
    $admin_id = $_SESSION['user']['id'];

    if (!empty($reply_text)) {
        $stmt = $pdo->prepare("
            INSERT INTO comment_replies (comment_id, admin_id, reply_text, created_at)
            VALUES (:comment_id, :admin_id, :reply_text, NOW())
        ");
        $stmt->execute([
            'comment_id' => $comment_id,
            'admin_id' => $admin_id,
            'reply_text' => $reply_text
        ]);

        $lastId = $pdo->lastInsertId();
        $replyStmt = $pdo->prepare("SELECT reply_text, created_at FROM comment_replies WHERE id = :id");
        $replyStmt->execute(['id' => $lastId]);
        $reply = $replyStmt->fetch(PDO::FETCH_ASSOC);

        echo json_encode($reply);
        exit;
    }
}

// Fetch all comments
$comments = $pdo->query("
    SELECT vc.*, u.fullname, c.name AS course_name
    FROM video_comments vc
    INNER JOIN users u ON u.id = vc.user_id
    INNER JOIN courses c ON c.id = vc.course_id
    ORDER BY vc.created_at DESC
")->fetchAll(PDO::FETCH_ASSOC);

// Fetch all replies
$repliesRaw = $pdo->query("SELECT * FROM comment_replies ORDER BY created_at ASC")->fetchAll(PDO::FETCH_ASSOC);

// Organize replies by comment
$repliesByComment = [];
foreach ($repliesRaw as $r) {
    $repliesByComment[$r['comment_id']][] = $r;
}
?>

<div class="container mt-4">
    <h4>💬 Comments Moderation & Replies</h4>

    <table class="table table-striped table-hover">
        <thead class="table-dark">
            <tr>
                <th>ID</th>
                <th>User</th>
                <th>Course</th>
                <th>Comment</th>
                <th>Created At</th>
                <th>Replies</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($comments as $c): ?>
            <tr>
                <td><?= $c['id'] ?></td>
                <td><?= htmlspecialchars($c['fullname']) ?></td>
                <td><?= htmlspecialchars($c['course_name']) ?></td>
                <td><?= htmlspecialchars($c['comment']) ?></td>
                <td><?= $c['created_at'] ?></td>
                <td>
                    <!-- Collapsible replies -->
                    <button class="btn btn-sm btn-outline-primary mb-1" data-bs-toggle="collapse" data-bs-target="#replies-<?= $c['id'] ?>">Show Replies</button>
                    <div class="collapse" id="replies-<?= $c['id'] ?>">
                        <ul class="list-unstyled" id="replies-list-<?= $c['id'] ?>">
                            <?php if (!empty($repliesByComment[$c['id']])): ?>
                                <?php foreach ($repliesByComment[$c['id']] as $r): ?>
                                    <li><small class="text-muted"><?= htmlspecialchars($r['reply_text']) ?> <em>(<?= $r['created_at'] ?>)</em></small></li>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <li><small class="text-muted">No replies yet.</small></li>
                            <?php endif; ?>
                        </ul>

                        <!-- AJAX Reply Form -->
                        <div class="input-group mt-2">
                            <input type="text" class="form-control form-control-sm reply-input" placeholder="Write a reply..." data-comment-id="<?= $c['id'] ?>" required>
                            <button class="btn btn-sm btn-primary reply-btn" data-comment-id="<?= $c['id'] ?>">Reply</button>
                        </div>
                    </div>
                </td>
                <td>
                    <a href="delete.php?type=comment&id=<?= $c['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Delete comment?')">Delete</a>
                </td>
            </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
</div>

<!-- JS -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
$(document).ready(function(){
    $('.reply-btn').click(function(e){
        e.preventDefault();
        var commentId = $(this).data('comment-id');
        var input = $('.reply-input[data-comment-id="'+commentId+'"]');
        var replyText = input.val().trim();
        if(replyText === '') return;

        $.post('', {
            ajax: true,
            comment_id: commentId,
            reply_text: replyText
        }, function(response){
            var reply = JSON.parse(response);
            var html = '<li><small class="text-muted">' + reply.reply_text + ' <em>(' + reply.created_at + ')</em></small></li>';
            $('#replies-list-'+commentId).append(html);
            input.val('');
            // Expand replies after adding
            var collapseEl = document.getElementById('replies-'+commentId);
            var bsCollapse = new bootstrap.Collapse(collapseEl, {show:true});
        });
    });
});
</script>
